"""
Configuration Loader - Loads and validates agent configuration
Supports loading from .env file, environment variables, and config.yaml
"""
import os
import uuid
import logging
from pathlib import Path
from typing import Optional, Dict, Any
from dataclasses import dataclass, field

import yaml

# Try to load dotenv for .env file support
try:
    from dotenv import load_dotenv
    # Load .env file from current directory
    env_path = Path('.env')
    if env_path.exists():
        load_dotenv(env_path)
except ImportError:
    pass  # dotenv not installed, use environment variables only

logger = logging.getLogger(__name__)


@dataclass
class AgentMetadata:
    """Agent metadata for identification and linking"""
    gpu_type: str = ""
    datacenter_id: str = ""
    rack_id: str = ""
    server_id: str = ""


@dataclass
class AgentConfig:
    """Agent configuration settings"""
    id: str
    name: str
    api_key: str
    api_endpoint: str
    interval_seconds: int = 30
    timeout_seconds: int = 10
    metadata: AgentMetadata = field(default_factory=AgentMetadata)


@dataclass
class SLAConfig:
    """SLA protection settings"""
    max_performance_loss_percent: float = 5.0
    grace_period_seconds: int = 60
    auto_revert: bool = True
    min_fps_threshold: Optional[float] = None


@dataclass
class GPUProfile:
    """GPU power profile"""
    id: str
    power_limit_percent: int
    clock_limit_mhz: Optional[int]
    description: str


@dataclass
class GPUConfig:
    """GPU configuration"""
    default_profile: str = "balanced"
    profiles: Dict[str, GPUProfile] = field(default_factory=dict)


@dataclass
class LoggingConfig:
    """Logging configuration"""
    level: str = "INFO"
    file: str = "logs/agent.log"
    max_size_mb: int = 10
    backup_count: int = 5


@dataclass
class NetworkConfig:
    """Network configuration"""
    verify_ssl: bool = True
    proxy: str = ""
    retry_attempts: int = 3
    retry_delay_seconds: int = 5


@dataclass
class Config:
    """Complete agent configuration"""
    agent: AgentConfig
    sla: SLAConfig
    gpu: GPUConfig
    logging: LoggingConfig
    network: NetworkConfig


def generate_agent_id() -> str:
    """Generate a unique agent ID"""
    import socket
    hostname = socket.gethostname()[:8].lower()
    short_uuid = str(uuid.uuid4())[:8]
    return f"agent-{hostname}-{short_uuid}"


def load_config(config_path: str = "config.yaml") -> Config:
    """
    Load configuration from YAML file.
    
    Args:
        config_path: Path to config file
        
    Returns:
        Config object
    """
    path = Path(config_path)
    
    if not path.exists():
        raise FileNotFoundError(f"Config file not found: {config_path}")
    
    with open(path, 'r') as f:
        raw = yaml.safe_load(f)
    
    # Parse agent config (prefer .env, then yaml, then defaults)
    agent_raw = raw.get('agent', {})
    agent_id = agent_raw.get('id') or os.getenv('ENERLASTIC_AGENT_ID') or generate_agent_id()
    api_key = os.getenv('ENERLASTIC_API_KEY') or agent_raw.get('api_key') or ''
    agent_name = os.getenv('AGENT_NAME') or agent_raw.get('name') or 'GPU Agent'
    
    # Load metadata from .env
    metadata = AgentMetadata(
        gpu_type=os.getenv('GPU_TYPE', ''),
        datacenter_id=os.getenv('DATACENTER_ID', ''),
        rack_id=os.getenv('RACK_ID', ''),
        server_id=os.getenv('SERVER_ID', ''),
    )
    
    agent = AgentConfig(
        id=agent_id,
        name=agent_name,
        api_key=api_key,
        api_endpoint=agent_raw.get('api_endpoint', 'https://app.enerlastic.com/api/optiflex/telemetry'),
        interval_seconds=agent_raw.get('interval_seconds', 30),
        timeout_seconds=agent_raw.get('timeout_seconds', 10),
        metadata=metadata,
    )
    
    # Parse SLA config
    sla_raw = raw.get('sla', {})
    sla = SLAConfig(
        max_performance_loss_percent=sla_raw.get('max_performance_loss_percent', 5.0),
        grace_period_seconds=sla_raw.get('grace_period_seconds', 60),
        auto_revert=sla_raw.get('auto_revert', True),
        min_fps_threshold=sla_raw.get('min_fps_threshold'),
    )
    
    # Parse GPU config
    gpu_raw = raw.get('gpu', {})
    profiles = {}
    for profile_id, profile_data in gpu_raw.get('profiles', {}).items():
        profiles[profile_id] = GPUProfile(
            id=profile_id,
            power_limit_percent=profile_data.get('power_limit_percent', 100),
            clock_limit_mhz=profile_data.get('clock_limit_mhz'),
            description=profile_data.get('description', ''),
        )
    
    gpu = GPUConfig(
        default_profile=gpu_raw.get('default_profile', 'balanced'),
        profiles=profiles,
    )
    
    # Parse logging config
    log_raw = raw.get('logging', {})
    logging_config = LoggingConfig(
        level=log_raw.get('level', 'INFO'),
        file=log_raw.get('file', 'logs/agent.log'),
        max_size_mb=log_raw.get('max_size_mb', 10),
        backup_count=log_raw.get('backup_count', 5),
    )
    
    # Parse network config
    net_raw = raw.get('network', {})
    network = NetworkConfig(
        verify_ssl=net_raw.get('verify_ssl', True),
        proxy=net_raw.get('proxy', ''),
        retry_attempts=net_raw.get('retry_attempts', 3),
        retry_delay_seconds=net_raw.get('retry_delay_seconds', 5),
    )
    
    config = Config(
        agent=agent,
        sla=sla,
        gpu=gpu,
        logging=logging_config,
        network=network,
    )
    
    # Log warnings
    if not config.agent.api_key:
        logger.warning("No API key configured - running in demo mode")
    
    return config


def setup_logging(config: LoggingConfig):
    """Setup logging based on configuration"""
    log_level = getattr(logging, config.level.upper(), logging.INFO)
    
    # Ensure log directory exists
    log_path = Path(config.file)
    log_path.parent.mkdir(parents=True, exist_ok=True)
    
    # Configure logging
    logging.basicConfig(
        level=log_level,
        format='%(asctime)s | %(levelname)-7s | %(message)s',
        datefmt='%Y-%m-%d %H:%M:%S',
        handlers=[
            logging.StreamHandler(),
            logging.FileHandler(config.file),
        ]
    )
