"""
Enerlastic GPU Agent - Main Entry Point

Real-time GPU telemetry and power optimization for Windows 11.
"""
import sys
import time
import signal
import logging
from pathlib import Path
from typing import Optional
import os

import click
import yaml

from .nvml_sampler import NVMLSampler, GPUTelemetry
from .power_controller import PowerController, DEFAULT_PROFILES
from .telemetry_publisher import TelemetryPublisher, APIResponse
from .sla_monitor import SLAMonitor, SLAConfig
from .config import load_config, setup_logging, Config

logger = logging.getLogger(__name__)


class EnerlasticAgent:
    """
    Main GPU Agent class that orchestrates telemetry collection,
    power optimization, and SLA monitoring.
    """
    
    def __init__(self, config: Config):
        """
        Initialize the agent.
        
        Args:
            config: Configuration object from config.py
        """
        self.config = config
        self._running = False
        
        # Initialize components
        self.sampler = NVMLSampler()
        
        self.publisher = TelemetryPublisher(
            api_endpoint=config.agent.api_endpoint,
            agent_id=config.agent.id,
            agent_name=config.agent.name,
            api_key=config.agent.api_key,
            timeout=config.agent.timeout_seconds,
            verify_ssl=config.network.verify_ssl,
            retry_attempts=config.network.retry_attempts,
        )
        
        # Controller will be initialized after NVML
        self.controller: Optional[PowerController] = None
        
        self.sla_monitor = SLAMonitor(SLAConfig(
            max_performance_loss_percent=config.sla.max_performance_loss_percent,
            grace_period_seconds=config.sla.grace_period_seconds,
            auto_revert=config.sla.auto_revert,
        ))
        
        self._interval = config.agent.interval_seconds
        self._default_profile = config.gpu.default_profile
        self._baseline_samples: list = []
    
    def start(self):
        """Start the agent"""
        logger.info("=" * 60)
        logger.info("Enerlastic GPU Agent Starting...")
        logger.info("=" * 60)
        
        # Initialize NVML
        if not self.sampler.initialize():
            logger.error("Failed to initialize NVML. Is NVIDIA driver installed?")
            return False
        
        gpu_count = self.sampler.get_gpu_count()
        logger.info(f"Detected {gpu_count} GPU(s)")
        
        # Get GPU info for registration
        sample = self.sampler.sample(0)
        if not sample:
            logger.error("Failed to sample GPU")
            return False
        
        logger.info(f"GPU 0: {sample.gpu_model}")
        logger.info(f"  TDP: {sample.power_limit_default_w}W")
        logger.info(f"  Power range: {sample.power_limit_min_w}W - {sample.power_limit_max_w}W")
        
        # Initialize power controller with actual TDP
        self.controller = PowerController(
            gpu_index=0,
            tdp_watts=int(sample.power_limit_default_w)
        )
        
        # Register with API
        logger.info(f"Connecting to API: {self.publisher.api_endpoint}")
        self.publisher.register(sample.gpu_model, gpu_count)
        
        # Collect baseline
        logger.info("Collecting baseline performance metrics...")
        self._collect_baseline()
        
        # Apply default profile
        if self._default_profile and self._default_profile != 'default':
            logger.info(f"Applying default profile: {self._default_profile}")
            if self.controller.apply_profile(self._default_profile):
                self.sla_monitor.on_profile_applied(self._default_profile)
            else:
                logger.warning("Failed to apply default profile")
        
        # Setup signal handlers
        signal.signal(signal.SIGINT, self._signal_handler)
        signal.signal(signal.SIGTERM, self._signal_handler)
        
        # Start main loop
        self._running = True
        logger.info(f"Agent started. Publishing every {self._interval}s")
        logger.info("-" * 60)
        
        self._main_loop()
        
        return True
    
    def stop(self):
        """Stop the agent"""
        logger.info("Stopping agent...")
        self._running = False
        
        # Revert to default settings
        if self.controller:
            self.controller.revert_to_default()
        
        # Shutdown NVML
        self.sampler.shutdown()
        
        logger.info("Agent stopped")
    
    def _signal_handler(self, signum, frame):
        """Handle shutdown signals"""
        logger.info(f"Received signal {signum}")
        self.stop()
        sys.exit(0)
    
    def _collect_baseline(self, sample_count: int = 5):
        """Collect baseline performance samples"""
        samples = []
        for i in range(sample_count):
            sample = self.sampler.sample(0)
            if sample:
                samples.append(sample)
            time.sleep(2)
        
        if samples:
            self._baseline_samples = samples
            self.sla_monitor.set_baseline(samples)
    
    def _main_loop(self):
        """Main agent loop"""
        while self._running:
            try:
                # Sample GPU telemetry
                sample = self.sampler.sample(0)
                if not sample:
                    logger.error("Failed to sample GPU")
                    time.sleep(self._interval)
                    continue
                
                # Add to SLA monitor
                self.sla_monitor.add_sample(sample)
                
                # Publish telemetry
                response = self.publisher.publish(sample)
                
                # Log status
                self._log_status(sample, response)
                
                # Handle commands from API
                if response.command:
                    self._handle_command(response)
                
                # Check SLA
                sla_status = self.sla_monitor.check_sla()
                if not sla_status['compliant']:
                    logger.warning(sla_status['message'])
                    if self.sla_monitor.should_revert():
                        logger.warning("Reverting to default profile due to SLA breach")
                        self.controller.revert_to_default()
                        self.sla_monitor.on_profile_applied('default')
                
                # Sleep until next sample
                time.sleep(self._interval)
                
            except Exception as e:
                logger.error(f"Error in main loop: {e}")
                time.sleep(self._interval)
    
    def _log_status(self, sample: GPUTelemetry, response: APIResponse):
        """Log current status"""
        profile = self.controller.get_current_profile() if self.controller else 'none'
        
        logger.info(
            f"[{sample.gpu_model}] "
            f"Power: {sample.power_draw_w:.0f}W/{sample.power_limit_w:.0f}W | "
            f"Clock: {sample.gpu_clock_mhz} MHz | "
            f"Util: {sample.gpu_utilization}% | "
            f"Temp: {sample.temperature_c}°C | "
            f"Profile: {profile}"
        )
        
        if not response.success:
            logger.warning(f"API: {response.message}")
    
    def _handle_command(self, response: APIResponse):
        """Handle optimization command from API"""
        if response.command == 'applyProfile' and response.profile:
            logger.info(f"Received command: Apply profile '{response.profile}'")
            if self.controller.apply_profile(response.profile):
                self.sla_monitor.on_profile_applied(response.profile)
            else:
                logger.error(f"Failed to apply profile: {response.profile}")
        
        elif response.command == 'revert':
            logger.info("Received command: Revert to defaults")
            self.controller.revert_to_default()
            self.sla_monitor.on_profile_applied('default')


@click.command()
@click.option('--config', '-c', default='config.yaml', help='Path to config file')
@click.option('--debug', is_flag=True, help='Enable debug logging')
def main(config: str, debug: bool):
    """Enerlastic GPU Agent for Windows 11"""
    # Load config
    try:
        cfg = load_config(config)
    except FileNotFoundError as e:
        print(f"ERROR: {e}")
        sys.exit(1)
    
    # Setup logging
    if debug:
        cfg.logging.level = 'DEBUG'
    setup_logging(cfg.logging)
    
    logger.info("=" * 60)
    logger.info("Enerlastic GPU Agent v1.0.0")
    logger.info("https://app.enerlastic.com")
    logger.info("=" * 60)
    
    # Create and start agent
    agent = EnerlasticAgent(cfg)
    
    try:
        if not agent.start():
            sys.exit(1)
    except KeyboardInterrupt:
        pass
    finally:
        agent.stop()


if __name__ == '__main__':
    main()
