"""
Power Controller - GPU Power Limit and Clock Control via nvidia-smi
"""
import subprocess
import logging
from typing import Optional, Dict, Any
from dataclasses import dataclass

logger = logging.getLogger(__name__)


@dataclass
class PowerProfile:
    """GPU Power Profile configuration"""
    id: str
    name: str
    power_limit_percent: int  # % of TDP
    clock_limit_mhz: Optional[int]
    description: str


# Default profiles for RTX 4090
DEFAULT_PROFILES: Dict[str, PowerProfile] = {
    'eco': PowerProfile(
        id='desktop-eco',
        name='Desktop Eco Mode',
        power_limit_percent=70,
        clock_limit_mhz=1800,
        description='Maximum efficiency - 30% power savings'
    ),
    'balanced': PowerProfile(
        id='desktop-balanced', 
        name='Desktop Balanced Mode',
        power_limit_percent=85,
        clock_limit_mhz=2100,
        description='Optimal balance - 20% power savings'
    ),
    'performance': PowerProfile(
        id='desktop-performance',
        name='Desktop Performance Mode',
        power_limit_percent=100,
        clock_limit_mhz=None,
        description='Full performance'
    ),
}


class PowerController:
    """
    GPU Power Controller using nvidia-smi commands.
    Controls power limits, clock frequencies, and fans.
    """
    
    def __init__(self, gpu_index: int = 0, tdp_watts: int = 450):
        """
        Initialize Power Controller.
        
        Args:
            gpu_index: GPU index (0-based)
            tdp_watts: GPU TDP in watts (450W for RTX 4090)
        """
        self.gpu_index = gpu_index
        self.tdp_watts = tdp_watts
        self._current_profile: Optional[str] = None
        self._original_power_limit: Optional[int] = None
    
    def _run_nvidia_smi(self, args: list) -> tuple[bool, str]:
        """
        Execute nvidia-smi command.
        
        Args:
            args: Command arguments
            
        Returns:
            Tuple of (success, output/error)
        """
        cmd = ['nvidia-smi', '-i', str(self.gpu_index)] + args
        
        try:
            result = subprocess.run(
                cmd,
                capture_output=True,
                text=True,
                timeout=10
            )
            
            if result.returncode == 0:
                return True, result.stdout.strip()
            else:
                return False, result.stderr.strip()
                
        except subprocess.TimeoutExpired:
            return False, "Command timed out"
        except FileNotFoundError:
            return False, "nvidia-smi not found in PATH"
        except Exception as e:
            return False, str(e)
    
    def get_current_power_limit(self) -> Optional[int]:
        """Get current power limit in watts"""
        success, output = self._run_nvidia_smi([
            '--query-gpu=power.limit',
            '--format=csv,noheader,nounits'
        ])
        
        if success:
            try:
                return int(float(output))
            except ValueError:
                pass
        return None
    
    def set_power_limit(self, watts: int) -> bool:
        """
        Set GPU power limit.
        
        Args:
            watts: Power limit in watts
            
        Returns:
            True if successful
        """
        # Store original for revert
        if self._original_power_limit is None:
            self._original_power_limit = self.get_current_power_limit()
        
        success, output = self._run_nvidia_smi(['-pl', str(watts)])
        
        if success:
            logger.info(f"GPU {self.gpu_index}: Power limit set to {watts}W")
            return True
        else:
            logger.error(f"Failed to set power limit: {output}")
            return False
    
    def set_power_limit_percent(self, percent: int) -> bool:
        """
        Set power limit as percentage of TDP.
        
        Args:
            percent: Power limit as % of TDP (e.g., 80 = 80%)
            
        Returns:
            True if successful
        """
        watts = int(self.tdp_watts * percent / 100)
        return self.set_power_limit(watts)
    
    def set_clock_limit(self, max_mhz: int) -> bool:
        """
        Set maximum GPU clock frequency.
        
        Args:
            max_mhz: Maximum clock in MHz
            
        Returns:
            True if successful
        """
        success, output = self._run_nvidia_smi([
            '-lgc', f'0,{max_mhz}'
        ])
        
        if success:
            logger.info(f"GPU {self.gpu_index}: Clock limit set to 0-{max_mhz} MHz")
            return True
        else:
            logger.error(f"Failed to set clock limit: {output}")
            return False
    
    def reset_clocks(self) -> bool:
        """Reset GPU clocks to default"""
        success, output = self._run_nvidia_smi(['-rgc'])
        
        if success:
            logger.info(f"GPU {self.gpu_index}: Clocks reset to default")
            return True
        else:
            logger.error(f"Failed to reset clocks: {output}")
            return False
    
    def apply_profile(self, profile_id: str) -> bool:
        """
        Apply a power profile.
        
        Args:
            profile_id: Profile ID (eco, balanced, performance)
            
        Returns:
            True if successful
        """
        if profile_id not in DEFAULT_PROFILES:
            logger.error(f"Unknown profile: {profile_id}")
            return False
        
        profile = DEFAULT_PROFILES[profile_id]
        logger.info(f"Applying profile: {profile.name}")
        
        # Set power limit
        success = self.set_power_limit_percent(profile.power_limit_percent)
        if not success:
            return False
        
        # Set clock limit if specified
        if profile.clock_limit_mhz:
            success = self.set_clock_limit(profile.clock_limit_mhz)
            if not success:
                return False
        else:
            # Reset to default clocks for performance mode
            self.reset_clocks()
        
        self._current_profile = profile_id
        logger.info(f"Profile applied: {profile.name} ({profile.description})")
        return True
    
    def revert_to_default(self) -> bool:
        """Revert to original power settings"""
        success = True
        
        # Reset clocks
        if not self.reset_clocks():
            success = False
        
        # Restore original power limit
        if self._original_power_limit:
            if not self.set_power_limit(self._original_power_limit):
                success = False
        
        self._current_profile = None
        logger.info("Reverted to default settings")
        return success
    
    def get_current_profile(self) -> Optional[str]:
        """Get currently active profile ID"""
        return self._current_profile
    
    def get_status(self) -> Dict[str, Any]:
        """Get current power controller status"""
        return {
            'gpuIndex': self.gpu_index,
            'tdpWatts': self.tdp_watts,
            'currentProfile': self._current_profile,
            'currentPowerLimit': self.get_current_power_limit(),
            'originalPowerLimit': self._original_power_limit,
        }


# Demo function
def demo_controller():
    """Demo power controller functions"""
    controller = PowerController(gpu_index=0, tdp_watts=450)
    
    print("GPU Power Controller Demo")
    print("-" * 40)
    
    # Get current status
    current = controller.get_current_power_limit()
    print(f"Current power limit: {current}W")
    
    # Apply balanced profile
    print("\nApplying balanced profile...")
    if controller.apply_profile('balanced'):
        print(f"New power limit: {controller.get_current_power_limit()}W")
    
    # Wait and revert
    import time
    time.sleep(5)
    
    print("\nReverting to defaults...")
    controller.revert_to_default()
    print(f"Power limit: {controller.get_current_power_limit()}W")


if __name__ == "__main__":
    logging.basicConfig(level=logging.INFO)
    demo_controller()
