"""
Telemetry Publisher - Send GPU telemetry to DeepOptiFlex API
"""
import json
import logging
import time
from typing import Optional, Dict, Any, List
from dataclasses import dataclass
import requests
from urllib3.util.retry import Retry
from requests.adapters import HTTPAdapter

from .nvml_sampler import GPUTelemetry

logger = logging.getLogger(__name__)


@dataclass
class APIResponse:
    """Response from DeepOptiFlex API"""
    success: bool
    message: str
    data: Optional[Dict[str, Any]] = None
    command: Optional[str] = None
    profile: Optional[str] = None


class TelemetryPublisher:
    """
    Publishes GPU telemetry to DeepOptiFlex API and receives optimization commands.
    Supports HTTPS with API key authentication.
    """
    
    def __init__(
        self,
        api_endpoint: str,
        agent_id: str,
        agent_name: str = "GPU Agent",
        api_key: str = "",
        timeout: int = 10,
        verify_ssl: bool = True,
        retry_attempts: int = 3,
    ):
        """
        Initialize Telemetry Publisher.
        
        Args:
            api_endpoint: DeepOptiFlex API endpoint URL (HTTPS)
            agent_id: Unique agent identifier
            agent_name: Human-readable agent name
            api_key: API key for authentication
            timeout: Request timeout in seconds
            verify_ssl: Whether to verify SSL certificates
            retry_attempts: Number of retry attempts on failure
        """
        self.api_endpoint = api_endpoint.rstrip('/')
        self.agent_id = agent_id
        self.agent_name = agent_name
        self.api_key = api_key
        self.timeout = timeout
        self.verify_ssl = verify_ssl
        
        # Setup session with retry
        self._session = requests.Session()
        
        # Retry strategy
        retry_strategy = Retry(
            total=retry_attempts,
            backoff_factor=1,
            status_forcelist=[429, 500, 502, 503, 504],
        )
        adapter = HTTPAdapter(max_retries=retry_strategy)
        self._session.mount("https://", adapter)
        self._session.mount("http://", adapter)
        
        # Set headers
        self._session.headers.update({
            'Content-Type': 'application/json',
            'User-Agent': 'EnerlasticAgent/1.0',
            'X-Agent-ID': agent_id,
            'X-Agent-Name': agent_name,
        })
        
        # Add API key header if provided
        if api_key:
            self._session.headers['Authorization'] = f'Bearer {api_key}'
            self._session.headers['X-API-Key'] = api_key
        
        self._last_publish_time: Optional[float] = None
        self._publish_count = 0
        self._error_count = 0
        
        # Log connection info
        logger.info(f"Publisher initialized: {self.api_endpoint}")
        if api_key:
            logger.info(f"API Key: {api_key[:8]}..." if len(api_key) > 8 else "API Key: (short key)")
        else:
            logger.warning("No API key - running in demo mode")
    
    def publish(self, telemetry: GPUTelemetry) -> APIResponse:
        """
        Publish single telemetry sample to API.
        
        Args:
            telemetry: GPU telemetry sample
            
        Returns:
            APIResponse with any commands from server
        """
        return self.publish_batch([telemetry])
    
    def publish_batch(self, samples: List[GPUTelemetry]) -> APIResponse:
        """
        Publish batch of telemetry samples to API.
        
        Args:
            samples: List of GPU telemetry samples
            
        Returns:
            APIResponse with any commands from server
        """
        if not samples:
            return APIResponse(success=False, message="No samples to publish")
        
        payload = {
            'agentId': self.agent_id,
            'agentName': self.agent_name,
            'hostname': self._get_hostname(),
            'timestamp': time.time(),
            'samples': [s.to_dict() for s in samples],
        }
        
        try:
            response = self._session.post(
                self.api_endpoint,
                json=payload,
                timeout=self.timeout
            )
            
            self._last_publish_time = time.time()
            self._publish_count += 1
            
            if response.status_code == 200:
                data = response.json()
                
                # Check for optimization commands
                command = data.get('command')
                profile = data.get('profile')
                
                if command:
                    logger.info(f"Received command: {command} (profile: {profile})")
                
                return APIResponse(
                    success=True,
                    message="Telemetry published",
                    data=data,
                    command=command,
                    profile=profile,
                )
            else:
                self._error_count += 1
                error_msg = f"API error: {response.status_code}"
                try:
                    error_msg = response.json().get('error', error_msg)
                except:
                    pass
                logger.error(error_msg)
                return APIResponse(success=False, message=error_msg)
                
        except requests.exceptions.Timeout:
            self._error_count += 1
            logger.error("API request timed out")
            return APIResponse(success=False, message="Request timed out")
            
        except requests.exceptions.ConnectionError:
            self._error_count += 1
            logger.error(f"Cannot connect to API: {self.api_endpoint}")
            return APIResponse(success=False, message="Connection failed")
            
        except Exception as e:
            self._error_count += 1
            logger.error(f"Publish error: {e}")
            return APIResponse(success=False, message=str(e))
    
    def register(self, gpu_model: str, gpu_count: int) -> bool:
        """
        Register agent with API.
        
        Args:
            gpu_model: GPU model name
            gpu_count: Number of GPUs
            
        Returns:
            True if registration successful
        """
        payload = {
            'action': 'register',
            'agentId': self.agent_id,
            'agentName': self.agent_name,
            'hostname': self._get_hostname(),
            'gpuModel': gpu_model,
            'gpuCount': gpu_count,
            'version': '1.0.0',
        }
        
        try:
            response = self._session.post(
                self.api_endpoint,
                json=payload,
                timeout=self.timeout
            )
            
            if response.status_code == 200:
                logger.info(f"Agent registered: {self.agent_id}")
                return True
            else:
                logger.error(f"Registration failed: {response.status_code}")
                return False
                
        except Exception as e:
            logger.error(f"Registration error: {e}")
            return False
    
    def heartbeat(self) -> bool:
        """Send heartbeat to maintain connection"""
        payload = {
            'action': 'heartbeat',
            'agentId': self.agent_id,
            'timestamp': time.time(),
        }
        
        try:
            response = self._session.post(
                self.api_endpoint,
                json=payload,
                timeout=5
            )
            return response.status_code == 200
        except:
            return False
    
    def get_stats(self) -> Dict[str, Any]:
        """Get publisher statistics"""
        return {
            'agentId': self.agent_id,
            'endpoint': self.api_endpoint,
            'publishCount': self._publish_count,
            'errorCount': self._error_count,
            'lastPublishTime': self._last_publish_time,
            'errorRate': (
                self._error_count / self._publish_count * 100
                if self._publish_count > 0 else 0
            ),
        }
    
    @staticmethod
    def _get_hostname() -> str:
        """Get system hostname"""
        import socket
        try:
            return socket.gethostname()
        except:
            return "unknown"


# Demo function
def demo_publisher():
    """Demo telemetry publisher"""
    from .nvml_sampler import NVMLSampler
    
    publisher = TelemetryPublisher(
        api_endpoint="http://localhost:3000/api/optiflex/telemetry",
        agent_id="rtx4090-demo-01",
        agent_name="RTX 4090 Demo Agent"
    )
    
    sampler = NVMLSampler()
    if not sampler.initialize():
        print("Failed to initialize NVML")
        return
    
    try:
        # Register agent
        sample = sampler.sample(0)
        if sample:
            publisher.register(sample.gpu_model, sampler.get_gpu_count())
        
        # Publish telemetry
        for i in range(5):
            samples = sampler.sample_all()
            response = publisher.publish_batch(samples)
            print(f"Publish {i+1}: {response.message}")
            if response.command:
                print(f"  Command: {response.command} -> {response.profile}")
            time.sleep(5)
        
        print("\nStats:", json.dumps(publisher.get_stats(), indent=2))
        
    finally:
        sampler.shutdown()


if __name__ == "__main__":
    logging.basicConfig(level=logging.INFO)
    demo_publisher()
