# Enerlastic GPU Agent Installation Script for Windows 11
# Run this script as Administrator!
# =====================================================

param(
    [switch]$SkipPython,
    [switch]$SkipTorch,
    [string]$ApiKey
)

$ErrorActionPreference = "Stop"
$ProgressPreference = "SilentlyContinue"

Write-Host ""
Write-Host "╔══════════════════════════════════════════════════════════╗" -ForegroundColor Cyan
Write-Host "║                                                          ║" -ForegroundColor Cyan
Write-Host "║           ENERLASTIC GPU AGENT INSTALLER                 ║" -ForegroundColor Cyan
Write-Host "║           GPU Power Optimization Platform                ║" -ForegroundColor Cyan
Write-Host "║                                                          ║" -ForegroundColor Cyan
Write-Host "╚══════════════════════════════════════════════════════════╝" -ForegroundColor Cyan
Write-Host ""

# Check if running as Administrator
$isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
if (-not $isAdmin) {
    Write-Host "❌ ERROR: This script must be run as Administrator!" -ForegroundColor Red
    Write-Host ""
    Write-Host "Right-click PowerShell and select 'Run as Administrator'" -ForegroundColor Yellow
    pause
    exit 1
}

Write-Host "✓ Running as Administrator" -ForegroundColor Green

# Check Python
Write-Host ""
Write-Host "Checking Python installation..." -ForegroundColor Yellow
try {
    $pythonVersion = python --version 2>&1
    Write-Host "✓ Python found: $pythonVersion" -ForegroundColor Green
} catch {
    Write-Host "❌ Python not found! Please install Python 3.10+" -ForegroundColor Red
    Write-Host "   Download from: https://www.python.org/downloads/" -ForegroundColor Yellow
    pause
    exit 1
}

# Check nvidia-smi
Write-Host ""
Write-Host "Checking NVIDIA GPU..." -ForegroundColor Yellow
try {
    $nvidiaSmi = nvidia-smi --query-gpu=name --format=csv,noheader 2>&1
    Write-Host "✓ GPU found: $nvidiaSmi" -ForegroundColor Green
} catch {
    Write-Host "❌ NVIDIA GPU not detected!" -ForegroundColor Red
    Write-Host "   Ensure NVIDIA drivers are installed." -ForegroundColor Yellow
    pause
    exit 1
}

# Install Python dependencies
if (-not $SkipPython) {
    Write-Host ""
    Write-Host "Installing Python dependencies..." -ForegroundColor Yellow
    pip install -r requirements.txt --quiet
    if ($LASTEXITCODE -eq 0) {
        Write-Host "✓ Dependencies installed" -ForegroundColor Green
    } else {
        Write-Host "⚠ Some dependencies may have failed" -ForegroundColor Yellow
    }
}

# Install PyTorch (optional for load simulator)
if (-not $SkipTorch) {
    Write-Host ""
    Write-Host "Installing PyTorch for load simulator (this may take a few minutes)..." -ForegroundColor Yellow
    pip install torch --index-url https://download.pytorch.org/whl/cu121 --quiet 2>$null
    if ($LASTEXITCODE -eq 0) {
        Write-Host "✓ PyTorch installed" -ForegroundColor Green
    } else {
        Write-Host "⚠ PyTorch installation failed (optional, simulator won't work)" -ForegroundColor Yellow
    }
}

# Configure API Key
Write-Host ""
Write-Host "═══════════════════════════════════════════════════════════" -ForegroundColor Cyan
Write-Host "                    API KEY SETUP                          " -ForegroundColor Cyan  
Write-Host "═══════════════════════════════════════════════════════════" -ForegroundColor Cyan
Write-Host ""

if ($ApiKey) {
    $apiKeyToUse = $ApiKey
} elseif ($env:ENERLASTIC_API_KEY) {
    Write-Host "Using API key from environment variable." -ForegroundColor Green
    $apiKeyToUse = $env:ENERLASTIC_API_KEY
} else {
    Write-Host "To connect to Enerlastic Pro, you need an API key." -ForegroundColor Yellow
    Write-Host ""
    Write-Host "Get your API key from: https://app.enerlastic.com/settings/api" -ForegroundColor Cyan
    Write-Host ""
    $apiKeyToUse = Read-Host "Enter your API key (or press Enter to skip)"
}

if ($apiKeyToUse) {
    # Update config.yaml with API key
    $configPath = Join-Path $PSScriptRoot "config.yaml"
    $configContent = Get-Content $configPath -Raw
    $configContent = $configContent -replace 'api_key:.*', "api_key: `"$apiKeyToUse`""
    Set-Content $configPath $configContent
    Write-Host "✓ API key configured" -ForegroundColor Green
    
    # Also set environment variable for current session
    $env:ENERLASTIC_API_KEY = $apiKeyToUse
} else {
    Write-Host "⚠ No API key provided. Agent will run in demo mode." -ForegroundColor Yellow
}

# Create start scripts
Write-Host ""
Write-Host "Creating start scripts..." -ForegroundColor Yellow

# Start Agent script
$startAgentScript = @"
@echo off
echo Starting Enerlastic GPU Agent...
echo.
python -m agent --config config.yaml
pause
"@
Set-Content "start-agent.bat" $startAgentScript

# Start Simulator script  
$startSimScript = @"
@echo off
echo Starting LLM Load Simulator...
echo.
set /p MODE="Select mode (training/inference/mixed) [mixed]: "
if "%MODE%"=="" set MODE=mixed
set /p DURATION="Duration in seconds [300]: "
if "%DURATION%"=="" set DURATION=300
echo.
python -m simulator.llm_load --mode %MODE% --duration %DURATION%
pause
"@
Set-Content "start-simulator.bat" $startSimScript

Write-Host "✓ Start scripts created" -ForegroundColor Green

# Done!
Write-Host ""
Write-Host "╔══════════════════════════════════════════════════════════╗" -ForegroundColor Green
Write-Host "║                                                          ║" -ForegroundColor Green
Write-Host "║           INSTALLATION COMPLETE! 🎉                      ║" -ForegroundColor Green
Write-Host "║                                                          ║" -ForegroundColor Green
Write-Host "╚══════════════════════════════════════════════════════════╝" -ForegroundColor Green
Write-Host ""
Write-Host "QUICK START:" -ForegroundColor Cyan
Write-Host ""
Write-Host "  1. Double-click 'start-agent.bat' to start the agent" -ForegroundColor White
Write-Host "  2. Double-click 'start-simulator.bat' to test with GPU load" -ForegroundColor White
Write-Host "  3. View results at https://app.enerlastic.com" -ForegroundColor White
Write-Host ""
Write-Host "Or run manually:" -ForegroundColor Yellow
Write-Host "  python -m agent --config config.yaml" -ForegroundColor Gray
Write-Host ""
Write-Host "For help: https://docs.enerlastic.com" -ForegroundColor Cyan
Write-Host ""

pause
